<?php
/**
 * Injects custom JavaScript into the WordPress footer for dynamic price calculations.
 *
 * This script ensures that product prices (regular and total) are recalculated dynamically
 * whenever the quantity input changes. It relies on the `wpc_after_scripts_loaded` action
 * and integrates seamlessly with Alpine.js and the WPC framework.
 *
 * - Updates the regular price based on the quantity.
 * - Updates the total price based on the quantity.
 * - Recalculates prices dynamically when the quantity input field is modified.
 */
add_action( 'wp_footer', function() {
    // Ensure the required action has been executed before injecting the script.
    if ( ! did_action( 'wpc_after_scripts_loaded' ) ) {
        return;
    }

    ?>
    <script type="text/javascript">
        (function() {
            'use strict';

            // Listen for the Alpine.js initialization event.
            document.addEventListener('alpine:initialized', () => {

                // Modify the regular price based on the quantity in the store.
                wpc.add_filter('wpc_regular_price', (regularPrice, { $store }) => {
                    return regularPrice * $store.quantity;
                });

                // Modify the total price based on the quantity in the store.
                wpc.add_filter('wpc_total_price', (totalPrice, { $store }) => {
                    return totalPrice * $store.quantity;
                });

                // Respond to changes in the quantity input field.
                wpc.add_action('wpc_quantity_input_changed', ({ configID, el }) => {
                    const $store = wpc.getStore(configID); // Retrieve the store for the given configuration ID.

                    if (el) {
                        $store.calculatePrice(); // Recalculate the price when the quantity changes.
                    }
                });

            });
        })();
    </script>
    <?php
}, 99);